import { IconCalendarEvent, IconMapPinFilled } from "@tabler/icons-react";
import { AdapterDayjs } from "@mui/x-date-pickers/AdapterDayjs";
import { LocalizationProvider } from "@mui/x-date-pickers/LocalizationProvider";
import { DateTimePicker } from "@mui/x-date-pickers/DateTimePicker";
import { useEffect, useState } from "react";
import { useDispatch, useSelector } from "react-redux";
import { Controller, useForm } from "react-hook-form";
import TextField from "@mui/material/TextField";
import { MenuItem } from "@mui/material";

//reducers
import { setAvailableCars, setLocationsOfDistrict, setSelectedDistrict } from "../../redux/user/selectRideSlice";

import * as z from "zod";
import { zodResolver } from "@hookform/resolvers/zod";
import { useNavigate } from "react-router-dom";
import { setSelectedData } from "../../redux/user/BookingDataSlice";
import dayjs from "dayjs";
import useFetchLocationsLov from "../../hooks/useFetchLocationsLov";

const schema = z.object({
  dropoff_location: z.string().min(1, { message: "Dropoff location needed" }),
  pickup_district: z.string().min(1, { message: "Pickup District needed" }),
  pickup_location: z.string().min(1, { message: "Pickup Location needed" }),

  pickuptime: z.object({
    $d: z.instanceof(Date).refine((date) => date !== null && date !== undefined, {
      message: "Date is not selected",
    }),
  }),

  dropofftime: z.object(
    {
      $L: z.string(), // Language code
      $d: z.date(), // Date object
      $y: z.number(), // Year
      $M: z.number(), // Month (0-indexed)
      $D: z.number(), // Day of month
      $W: z.number(), // Day of week (0-indexed, starting from Sunday)
      $H: z.number(), // Hour
      $m: z.number(), // Minute
      $s: z.number(), // Second
      $ms: z.number(), // Millisecond
      $isDayjsObject: z.boolean(), // Indicator for Day.js object
    },
    { message: "drop-off time is required" }
  ),
});

const CarSearch = () => {
  const {
    handleSubmit,
    control,
    reset,
    formState: { errors },
  } = useForm({
    resolver: zodResolver(schema),
    defaultValues: {
      pickup_district: "",
      pickup_location: "",
      dropoff_location: "",
      pickuptime: null,
      dropofftime: null,
    },
  });

  const navigate = useNavigate();
  const { districtData } = useSelector((state) => state.modelDataSlice);
  const { fetchLov, isLoading } = useFetchLocationsLov();
  const uniqueDistrict = districtData?.filter((cur, idx) => {
    return cur !== districtData[idx + 1];
  });
  const { selectedDistrict, wholeData, locationsOfDistrict } = useSelector((state) => state.selectRideSlice);

  const [pickup, setPickup] = useState(null);
  const [error, setError] = useState(null);

  const dispatch = useDispatch();

  //useEffect to fetch data from backend for locations
  useEffect(() => {
    // fetchModelData(dispatch);
    fetchLov();
  }, []);

  //for showing appropriate locations according to districts
  useEffect(() => {
    if (selectedDistrict !== null) {
      const showLocationInDistrict = wholeData
        .filter((cur) => {
          return cur.district === selectedDistrict;
        })
        .map((cur) => cur.location);
      dispatch(setLocationsOfDistrict(showLocationInDistrict));
    }
  }, [selectedDistrict]);

  //search cars
  const hanldeData = async (data) => {
    try {
      if (data) {
        //preserving the selected data for later use
        dispatch(setSelectedData(data));

        const pickupDate = data.pickuptime.$d;
        const dropOffDate = data.dropofftime.$d;
        const datas = {
          pickupDate,
          dropOffDate,
          pickUpDistrict: data.pickup_district,
          pickUpLocation: data.pickup_location,
        };

        const res = await fetch("api/user/showSingleofSameModel", {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify(datas),
        });

        if (!res.ok) {
          const data = await res.json();
          setError(data.message);
          return;
        }

        if (res.ok) {
          const result = await res.json();
          dispatch(setAvailableCars(result));
          navigate("/availableVehicles");
        }

        if (res.ok) {
          reset({
            pickuptime: null, // Reset pickuptime to null
            dropofftime: null, // Reset dropofftime to null
          });

          const pickupDistrictElement = document.getElementById("pickup_district");
          const pickupLocationElement = document.getElementById("pickup_location");
          const dropoffLocationElement = document.getElementById("dropoff_location");

          if (pickupDistrictElement) {
            pickupDistrictElement.innerHTML = "";
          }
          if (pickupLocationElement) {
            pickupLocationElement.innerHTML = "";
          }
          if (dropoffLocationElement) {
            dropoffLocationElement.innerHTML = "";
          }
        }
      }
    } catch (error) {
      console.log("Error  : ", error);
    }
  };

  //this is to ensure there will be 1 day gap between pickup and dropoff date

  const oneDayGap = pickup && pickup.add(1, "day");

  return (
    <>
      <section id="booking-section" className="relative z-10 py-20 mx-auto max-w-7xl px-6">
        <div className="text-center mb-12">
          <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
            Find Your <span className="bg-gradient-to-r from-indigo-500 to-purple-600 bg-clip-text text-transparent">Perfect Ride</span>
          </h2>
          <p className="text-gray-600 max-w-2xl mx-auto">
            Search from our wide range of vehicles available at your preferred location
          </p>
        </div>

        <div className="bg-white rounded-3xl shadow-card p-8 md:p-10 border border-gray-100">
          <form onSubmit={handleSubmit(hanldeData)}>
            <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
              {/* Pick-up District */}
              <div className="space-y-2">
                <label htmlFor="pickup_district" className="flex items-center gap-2 text-sm font-semibold text-gray-700">
                  <div className="w-8 h-8 bg-gradient-to-r from-indigo-500 to-purple-600 rounded-lg flex items-center justify-center">
                    <IconMapPinFilled className="w-4 h-4 text-white" />
                  </div>
                  Pick-up District <span className="text-red-500">*</span>
                </label>
                <Controller
                  name="pickup_district"
                  control={control}
                  render={({ field }) => (
                    <TextField
                      {...field}
                      id="pickup_district"
                      className="w-full"
                      select
                      size="small"
                      error={Boolean(errors.pickup_district)}
                      onChange={(e) => {
                        field.onChange(e.target.value);
                        dispatch(setSelectedDistrict(e.target.value));
                      }}
                      sx={{
                        '& .MuiOutlinedInput-root': {
                          borderRadius: '12px',
                          backgroundColor: '#f8fafc',
                          '&:hover': { backgroundColor: '#f1f5f9' },
                          '&.Mui-focused': { backgroundColor: '#fff' }
                        }
                      }}
                    >
                      {isLoading == true && (
                        <MenuItem value="">
                          <span className="animate-pulse">Loading...</span>
                        </MenuItem>
                      )}
                      {!isLoading && <MenuItem value="">Select a District</MenuItem>}
                      {uniqueDistrict?.map((cur, idx) => (
                        <MenuItem value={cur} key={idx} className="capitalize">
                          {cur}
                        </MenuItem>
                      ))}
                    </TextField>
                  )}
                />
                {errors.pickup_district && <p className="text-red-500 text-xs">{errors.pickup_district.message}</p>}
              </div>

              {/* Pick-up Location */}
              <div className="space-y-2">
                <label htmlFor="pickup_location" className="flex items-center gap-2 text-sm font-semibold text-gray-700">
                  <div className="w-8 h-8 bg-gradient-to-r from-indigo-500 to-purple-600 rounded-lg flex items-center justify-center">
                    <IconMapPinFilled className="w-4 h-4 text-white" />
                  </div>
                  Pick-up Location <span className="text-red-500">*</span>
                </label>
                <Controller
                  name="pickup_location"
                  control={control}
                  render={({ field }) => (
                    <TextField
                      {...field}
                      id="pickup_location"
                      select
                      size="small"
                      className="w-full capitalize"
                      onChange={(e) => field.onChange(e.target.value)}
                      error={Boolean(errors.pickup_location)}
                      sx={{
                        '& .MuiOutlinedInput-root': {
                          borderRadius: '12px',
                          backgroundColor: '#f8fafc',
                          '&:hover': { backgroundColor: '#f1f5f9' },
                          '&.Mui-focused': { backgroundColor: '#fff' }
                        }
                      }}
                    >
                      {isLoading && (
                        <MenuItem value="">
                          <span className="animate-pulse">Loading...</span>
                        </MenuItem>
                      )}
                      {!isLoading && <MenuItem value="">Select a Location</MenuItem>}
                      {locationsOfDistrict &&
                        locationsOfDistrict.map((availableLocations, idx) => (
                          <MenuItem value={availableLocations} key={idx}>
                            {availableLocations}
                          </MenuItem>
                        ))}
                    </TextField>
                  )}
                />
                {errors.pickup_location && <p className="text-red-500 text-xs">{errors.pickup_location.message}</p>}
              </div>

              {/* Drop-off Location */}
              <div className="space-y-2">
                <label className="flex items-center gap-2 text-sm font-semibold text-gray-700">
                  <div className="w-8 h-8 bg-gradient-to-r from-indigo-500 to-purple-600 rounded-lg flex items-center justify-center">
                    <IconMapPinFilled className="w-4 h-4 text-white" />
                  </div>
                  Drop-off Location <span className="text-red-500">*</span>
                </label>
                <Controller
                  name="dropoff_location"
                  control={control}
                  render={({ field }) => (
                    <TextField
                      {...field}
                      select
                      size="small"
                      error={Boolean(errors.dropoff_location)}
                      id="dropoff_location"
                      className="w-full capitalize"
                      onChange={(e) => field.onChange(e.target.value)}
                      sx={{
                        '& .MuiOutlinedInput-root': {
                          borderRadius: '12px',
                          backgroundColor: '#f8fafc',
                          '&:hover': { backgroundColor: '#f1f5f9' },
                          '&.Mui-focused': { backgroundColor: '#fff' }
                        }
                      }}
                    >
                      {isLoading && (
                        <MenuItem value="">
                          <span className="animate-pulse">Loading...</span>
                        </MenuItem>
                      )}
                      {!isLoading && <MenuItem value="">Select a Location</MenuItem>}
                      {locationsOfDistrict &&
                        locationsOfDistrict.map((availableLocations, idx) => (
                          <MenuItem value={availableLocations} key={idx}>
                            {availableLocations}
                          </MenuItem>
                        ))}
                    </TextField>
                  )}
                />
                {errors.dropoff_location && <p className="text-red-500 text-xs">{errors.dropoff_location.message}</p>}
              </div>

              {/* Pick-up Date */}
              <div className="space-y-2">
                <label htmlFor="picktime" className="flex items-center gap-2 text-sm font-semibold text-gray-700">
                  <div className="w-8 h-8 bg-gradient-to-r from-indigo-500 to-purple-600 rounded-lg flex items-center justify-center">
                    <IconCalendarEvent className="w-4 h-4 text-white" />
                  </div>
                  Pick-up Date <span className="text-red-500">*</span>
                </label>
                <Controller
                  name={"pickuptime"}
                  control={control}
                  render={({ field }) => (
                    <LocalizationProvider dateAdapter={AdapterDayjs}>
                      <DateTimePicker
                        label="Select pickup time"
                        {...field}
                        value={field.value}
                        minDate={dayjs()}
                        onChange={(newValue) => {
                          field.onChange(newValue);
                          setPickup(newValue);
                        }}
                        slotProps={{
                          textField: {
                            size: 'small',
                            fullWidth: true,
                            sx: {
                              '& .MuiOutlinedInput-root': {
                                borderRadius: '12px',
                                backgroundColor: '#f8fafc',
                                '&:hover': { backgroundColor: '#f1f5f9' },
                                '&.Mui-focused': { backgroundColor: '#fff' }
                              }
                            }
                          }
                        }}
                      />
                    </LocalizationProvider>
                  )}
                />
                {errors.pickuptime && <p className="text-red-500 text-xs">{errors.pickuptime.message}</p>}
              </div>

              {/* Drop-off Date */}
              <div className="space-y-2">
                <label htmlFor="droptime" className="flex items-center gap-2 text-sm font-semibold text-gray-700">
                  <div className="w-8 h-8 bg-gradient-to-r from-indigo-500 to-purple-600 rounded-lg flex items-center justify-center">
                    <IconCalendarEvent className="w-4 h-4 text-white" />
                  </div>
                  Drop-off Date <span className="text-red-500">*</span>
                </label>
                <Controller
                  name={"dropofftime"}
                  control={control}
                  render={({ field }) => (
                    <LocalizationProvider dateAdapter={AdapterDayjs}>
                      <DateTimePicker 
                        label="Select dropoff time" 
                        {...field} 
                        value={field.value} 
                        minDate={pickup ? oneDayGap : dayjs()}
                        slotProps={{
                          textField: {
                            size: 'small',
                            fullWidth: true,
                            sx: {
                              '& .MuiOutlinedInput-root': {
                                borderRadius: '12px',
                                backgroundColor: '#f8fafc',
                                '&:hover': { backgroundColor: '#f1f5f9' },
                                '&.Mui-focused': { backgroundColor: '#fff' }
                              }
                            }
                          }
                        }}
                      />
                    </LocalizationProvider>
                  )}
                />
                {errors.dropofftime && <p className="text-red-500 text-xs">{errors.dropofftime.message}</p>}
                {error && <p className="text-xs text-red-500">{error}</p>}
              </div>

              {/* Search Button */}
              <div className="flex items-end">
                <button 
                  type="submit" 
                  className="w-full py-3.5 bg-gradient-to-r from-indigo-500 to-purple-600 text-white font-semibold rounded-xl shadow-lg shadow-indigo-500/30 hover:shadow-xl hover:shadow-indigo-500/40 transition-all duration-300 hover:-translate-y-0.5 flex items-center justify-center gap-2"
                >
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                  </svg>
                  Search Cars
                </button>
              </div>
            </div>
          </form>
        </div>
      </section>
    </>
  );
};

export default CarSearch;
